/* wconfig_kvp_pool.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2025 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "wconfig_kvp_pool.hh"
#include "datei.h"

bool WConfigKVPPool::kvp_entry::operator==( const kvp_entry &rhs ) const
{
    return type == rhs.type &&
        bool_value == rhs.bool_value &&
        int_value == rhs.int_value &&
        str_value == rhs.str_value;
}

WConfigKVPPool::WConfigKVPPool()
{
}

WConfigKVPPool::get_entry_result WConfigKVPPool::getEntry( const std::string &key ) const
{
    auto default_it = s_all_entries.find( key );

    if ( default_it == s_all_entries.end() ) {
        return { .value = false,
                 .default_value = false,
                 .found = false,
                 .overwritten = false };
    }

    auto entry_it = m_entries.find( key );

    if ( entry_it == m_entries.end() ) {
        return { .value = default_it->second,
                 .default_value = default_it->second,
                 .found = true,
                 .overwritten = false };
    }

    if ( entry_it->second == default_it->second ) {
        return { .value = entry_it->second,
                 .default_value = default_it->second,
                 .found = true,
                 .overwritten = false };
    } else {
        return { .value = entry_it->second,
                 .default_value = default_it->second,
                 .found = true,
                 .overwritten = true };
    }
}

void WConfigKVPPool::setEntry( const std::string &key,
                               kvp_entry value )
{
    m_entries[key] = value;
}

std::set< std::string > WConfigKVPPool::getKeys() const
{
    std::set< std::string > res;

    for ( const auto &e : s_all_entries ) {
        res.insert( e.first );
    }

    for ( const auto &e : m_entries ) {
        res.insert( e.first );
    }

    return res;
}

void WConfigKVPPool::save( Datei *fh ) const
{
    fh->configOpenSection( "kvpentries" );

    for ( const auto &e : m_entries ) {
        if ( ! getEntry( e.first ).overwritten ) continue;

        fh->configOpenSection( "kvpentry" );

        fh->configPutPairString( "name", e.first.c_str() );
        switch ( e.second.type ) {
            case kvp_entry::KVP_BOOLEAN:
                fh->configPutPairString( "type", "bool" );
                fh->configPutPairBool( "value", e.second.bool_value );
                break;
            case kvp_entry::KVP_INT:
                fh->configPutPairString( "type", "int" );
                fh->configPutPairNum( "value", e.second.int_value );
                break;
            case kvp_entry::KVP_STRING:
                fh->configPutPairString( "type", "string" );
                fh->configPutPairString( "value", e.second.str_value.c_str() );
                break;
        }
        fh->configCloseSection();
    }

    fh->configCloseSection();
}

void WConfigKVPPool::clear()
{
}
